#include <Arduino_GFX_Library.h>
#include <ArduinoJson.h>
#include "gui.h"

// FOR ARDUINO Uncomment the line below if you wish debug messages
// #define CORE_DEBUG_LEVEL 4

// FOR VSCODE see platformio.ini file -D CORE_DEBUG_LEVEL=1

/***************************************************************************************************
 * Arduino_GFX Setup for JC4827W543C
 ***************************************************************************************************/
#define SCREEN_WIDTH 480
#define SCREEN_HEIGHT 272
#define SCR_BUF_LEN 32

// LCD backlight PWM
#define LCD_BL 1			 // lcd BL pin
#define LEDC_CHANNEL_0 0	 // use first channel of 16 channels (started from zero)
#define LEDC_TIMER_12_BIT 12 // use 12 bit precission for LEDC timer
#define LEDC_BASE_FREQ 5000	 // use 5000 Hz as a LEDC base frequency

Arduino_DataBus *bus = new Arduino_ESP32QSPI(
	45 /* cs */, 47 /* sck */, 21 /* d0 */, 48 /* d1 */, 40 /* d2 */, 39 /* d3 */);
Arduino_NV3041A *panel = new Arduino_NV3041A(bus, GFX_NOT_DEFINED /* RST */, 0 /* rotation */, true /* IPS */);
Arduino_GFX *gfx = new Arduino_Canvas(SCREEN_WIDTH /* width */, SCREEN_HEIGHT /* height */, panel);
//------------------------------- End TFT+TOUCH setup ------------------------------

static const char *TAG = "main";

/***************************************************************************************************
 * LVGL functions
 ***************************************************************************************************/

//=====================================================================================================================
void IRAM_ATTR disp_flush(lv_disp_drv_t *disp, const lv_area_t *area, lv_color_t *color_p)
{
	uint32_t w = (area->x2 - area->x1 + 1);
	uint32_t h = (area->y2 - area->y1 + 1);

	panel->startWrite();
	panel->setAddrWindow(area->x1, area->y1, w, h);
	panel->writePixels((uint16_t *)&color_p->full, w * h);
	panel->endWrite();
	lv_disp_flush_ready(disp);
}

//=====================================================================================================================
// value has to be between 0 and 255
void setBrightness(uint8_t value)
{
	uint32_t duty = 4095 * value / 255;
	ledcWrite(LCD_BL, duty);
}

//=====================================================================================================================

/***************************************************************************************************
 * LVGL Setup
 ***************************************************************************************************/

//=====================================================================================================================
void lvgl_init()
{
	static lv_disp_draw_buf_t draw_buf;
	static lv_disp_drv_t disp_drv;
	static lv_color_t disp_draw_buf[SCREEN_WIDTH * SCR_BUF_LEN];
	// static lv_color_t disp_draw_buf2[SCREEN_WIDTH * SCR_BUF_LEN];

	// Setting up the LEDC and configuring the Back light pin
	ledcAttach(LCD_BL, LEDC_BASE_FREQ, LEDC_TIMER_12_BIT);
	setBrightness(255);

	// Init Display
	if (!gfx->begin()) {
		ESP_LOGI(TAG, "gfx->begin() failed!\n");
	}
	gfx->fillScreen(BLACK);

	lv_init();

	if (!disp_draw_buf)
		ESP_LOGE(TAG, "LVGL disp_draw_buf allocate failed!");
	else {
		lv_disp_draw_buf_init(&draw_buf, disp_draw_buf, NULL, SCREEN_WIDTH * SCR_BUF_LEN);

		// Initialize the display
		lv_disp_drv_init(&disp_drv);
		disp_drv.hor_res = SCREEN_WIDTH;
		disp_drv.ver_res = SCREEN_HEIGHT;
		disp_drv.flush_cb = disp_flush;
		disp_drv.draw_buf = &draw_buf;
		lv_disp_drv_register(&disp_drv);
	}

	ESP_LOGI(TAG, "Lvgl v%d.%d.%d initialized\n", lv_version_major(), lv_version_minor(), lv_version_patch());
}

//=====================================================================================================================

INFO     	Info;

//=====================================================================================================================
void setup()
{
	Serial.begin(115200);
	lvgl_init();
	gui_init();
	ESP_LOGI(TAG, "Setup done");
}

StaticJsonDocument<512> doc; 

//=====================================================================================================================
bool serial_parse_json(const char* str)
{
	DeserializationError err = deserializeJson(doc, str);
	if (err == DeserializationError::Ok) {
		Info.Updated = millis();		
		float credit = doc["credit"] | NAN;
		if (!isnan(credit)) {
			Info.Credit = credit;
		}
		float price = doc["price"] | NAN;
		if (!isnan(price)) {
			Info.Price = price;
		}
		const char* message = doc["message"];
		if (message != NULL) {
			float price = doc["time"] | 2;
			gui_add_message(message, int(price * 1000));
		}
		const char* command = doc["command"];
		if (command != NULL && strcmp(command, "whois") == 0) {
			HWCDCSerial.write("{\"device\":\"display\"}\n");
		}
		return true;
	}
	//HWCDCSerial.printf("Err: %s [%s]\n", err.c_str(), str);
	return false;
}

//=====================================================================================================================
void serial_parse()
{
	static char buf[512];
	static int size = 0;
	int rx_count = HWCDCSerial.available();
	if (rx_count) {
		int left = sizeof(buf) - rx_count - 1;
		if (rx_count > left) rx_count = left;
		HWCDCSerial.read(buf + size, rx_count);
		size += rx_count;
		buf[size] = 0;
		//HWCDCSerial.printf("Rx: [%s]\n", buf);
		char* pos = buf;
		char* end;
		while ((end = strchr(pos, '\n')) != NULL && pos < buf + size) {
			*end = 0;
			serial_parse_json(pos);
			pos = end + 1;
		}
		if (pos < buf + size) {
			if (serial_parse_json(pos))
				size = 0;
			else if (pos != buf) {
				size -= pos - buf;
				memcpy(buf, pos, size);
			}
		} else
			size = 0;
	}
	if (size >= sizeof(buf)) size = 0;
}

//=====================================================================================================================
void loop()
{
	serial_parse();
	gui_update();
	lv_timer_handler();
	delay(5);
}

//=====================================================================================================================
