#!/usr/bin/python3

import logging
import time
from enum import Enum
from typing import List, Dict, Optional
import json
from .serialdev.device import Device
from .dataport import DataPort

logger = logging.getLogger("coral")

class Coral:

	def __init__(self, control_port: str, data_port: str, udp_port: int, baudrate: int = 115200) -> None:
		self.data_port = data_port
		self.udp_port = udp_port
		self.control = Device(control_port)
		self.control.open()
		self.control.set_baudrate(baudrate)
		self.control.read_all()
		#for addr, reg in self.control.registers.items():
		#	logger.debug("{}: {} = {}".format(reg.name, reg.type.name, str(reg)))

		if data_port is not None:
			self.data = DataPort(data_port, udp_port)
		else:
			self.data = DataPort(self.control.port, udp_port)
		#logger.info(f"info: {self.get_info()}")
		#logger.info(f"settings: {self.get_settings()}")

	def get_info(self) -> str:
		device = self.control["Устройство"]
		if device == "Коралл-15":
			freq_min = 130.0
			freq_max = 180.0
		else:
			freq_min = 300.0
			freq_max = 2800.0
		info = dict(
			device = "transmitter",
			type = "control",
			name = self.control["Устройство"],
			version = self.control["Версия"],
			frequency = [ freq_min, freq_max ],
		)
		return info

	def get_settings(self) -> str:
		if self.control["Использовать ППРЧ"]:
			settings = dict(
				seed = self.control["Зерно"],
				frequency = [ self.control["Диапазон частот от"], self.control["Диапазон частот до"] ],
				frequency_step = self.control["Шаг частоты"],
				power_amp = self.control["Мощность усилителя"]
			)
		else:
			settings = dict(
				seed = self.control["Зерно"],
				frequency = self.control["Диапазон частот от"],
				power_amp = self.control["Мощность усилителя"]
			)
		return settings

	def set_settings(self, values: dict) -> str:
		integral_result = True
		for index, value in values.items():
			result = self.settings(index, value)
			if not result:
				integral_result = False
		return integral_result

	def get_status(self) -> str:
		state = dict(
			state = self.control["Состояние"],
			crsf_rate = self.control["CrossFire пакетов каналов в секунду"],
			seed = self.control["Зерно"],
			packet_rate = 31.25,
			power_amp = self.control["Мощность усилителя"]
		)
		if self.control["Использовать ППРЧ"]:
			state["frequency"] = [ self.control["Диапазон частот от"], self.control["Диапазон частот до"] ]
			state["frequency_step"] = self.control["Шаг частоты"],
		else:
			state["frequency"] = self.control["Диапазон частот от"],
		return state

	def send_channels(self, payload: dict):
		return self.data.send_channels(payload)

	def command(self, cmd: str) -> bool:
		if cmd == "bind":
			try:
				self.control["Состояние"] = "BINDING"
				return True
			except Exception:
				return False
		else:
			return False

	def settings(self, name: str, value) -> bool:
		if name == "seed":
			result = self.control.write("Зерно", str(value))
			logger.info(f"Set {name}={value}, result {result}")
			return result
		elif name == "power":
			result = self.control.write("Мощность усилителя", int(value))
			logger.info(f"Set {name}={value}, result {result}")
			return result
		elif name == "frequency":
			if type(value) is str:
				value = list(map(float, value.strip('()[]').split(',')))
			if type(value) is list:
				if len(value) > 1:
					result_1 = self.control.write("Диапазон частот от", float(value[0]))
					result_2 = self.control.write("Диапазон частот до", float(value[1]))
					result_3 = self.control.write("Использовать ППРЧ", 1)
					result = result_1 and result_2 and result_3
					logger.info(f"Set {name}=[{value[0]},{value[1]}], result {result}")
				else:
					result_1 = self.control.write("Диапазон частот от", float(value[0]))
					result_3 = self.control.write("Использовать ППРЧ", 0)
					result = result_1 and result_2
					logger.info(f"Set {name}={value[0]}, result {result}")
			else:
				result = False
			return result
		elif name == "packet_rate":
			result = (value >= 30) and (value <= 32)
			logger.info(f"Set {name}={value}, result {result}")
			return result
		else:
			return False

	def close(self):
		self.control.close()
		self.data.close()

	def __getitem__(self, index):
		return self.control[index]

	def __setitem__(self, index, value):
		self.control[index] = value

if __name__ == '__main__':
	print("start")
	
	gs_conf = dotenv_values("/gs/config/gs.conf")
	control_port = "/dev/ttyS6"
	data_port = "/dev/ttyS2"
	udp_port = 169

	coral = Coral(control_port, data_port, udp_port)

	while True:
		time.sleep(1)
