import logging
import time
import threading
from typing import List, Dict, Optional
from serial import Serial, EIGHTBITS, PARITY_NONE, STOPBITS_ONE, SerialException
import socket
from queue import Queue, Empty
from .crsf.payloads import PacketsTypes
from .crsf.frames import crsf_frame
from .crsf.handling import crsf_build_frame

logger = logging.getLogger("coral.data")

def map_range(x, in_min=0, in_max=2048, out_min=173, out_max=1812):
	return int(out_min + (x - in_min) * (out_max - out_min) / (in_max - in_min))

def build_packet(payload: dict) -> bytes:
	axes = payload.get("axes")
	buttons = payload.get("buttons")
	channels = payload.get("channels")
	if channels is not None:
		channels = (channels + [0] * 16)[:16]
		logger.debug(f"Channels: {channels}")
		frame = crsf_build_frame(
			PacketsTypes.RC_CHANNELS_PACKED,
			{"channels": channels}
		)
		return frame
	elif axes is not None and buttons is not None:
		channels = list(map(lambda x: map_range(x), axes))
		buttons = list(map(lambda x: 1811 if x == 1 else 173, buttons))
		to_send = [0] * 16
		to_send[:8] = (channels + [173] * 8)[:8]
		to_send[8:] = (buttons + [173] * 8)[:8]
		logger.debug(f"Channels: {to_send}")
		frame = crsf_build_frame(
			PacketsTypes.RC_CHANNELS_PACKED,
			{"channels": to_send}
		)
		return frame
	else:
		return None

class DataPort:
	serial_port_name: str
	port: Serial = None
	packets = Queue()
	
	def __init__(self, serial_port: str | Serial, udp_port: int | None, log_level=logging.INFO) -> None:
		if type(serial_port) is Serial:
			self.port = serial_port
			self.serial_port_name = None
		else:
			self.serial_port_name = serial_port
		self.udp_port = int(udp_port) if udp_port is not None else None
		logger.setLevel(log_level)

		self.stop_request = threading.Event()
		if self.udp_port is not None:
			self.udp_loop_thread = threading.Thread(target=self.udp_loop, daemon=True, name="coral_data_udp")
			self.udp_loop_thread.start()
		self.serial_loop_thread = threading.Thread(target=self.serial_loop, daemon=True, name="coral_data_serial")
		self.serial_loop_thread.start()

	def open(self) -> bool:
		self.port = Serial()
		self.port.port = self.serial_port_name
		self.port.baudrate = 420000
		self.port.bytesize = EIGHTBITS
		self.port.parity = PARITY_NONE
		self.port.stopbits = STOPBITS_ONE
		self.port.timeout = 0.1
		self.port.inter_byte_timeout = 0.01
		self.port.xonxoff = False
		self.port.rtscts = False
		self.port.dsrdtr = False
		try: 
			self.port.open()
			logger.info(f"Serial port {self.serial_port_name} opened")
		except Exception as e:
			logger.error(f"Opening serial port {self.serial_port_name}: {e}")
			return False
		self.port.flushInput()
		self.port.flushOutput()
		return True
	
	def opened(self) -> bool:
		if self.port is not None:
			return self.port.is_open
		else:
			return False
	
	def close(self):
		if self.opened():
			self.port.close()
			self.port = None
	
	def udp_loop(self):
		sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
		sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
		sock.bind(("0.0.0.0", self.udp_port))
		logger.info(f"Listen UDP {self.udp_port} for CRSF or JSON")

		while not self.stop_request.is_set():
			try:
				data, addr = sock.recvfrom(1024)
				decoded_data = data.decode('utf-8')

				try:
					obj = json.loads(decoded_data)
					self.packets.put(build_packet(obj))
				except json.JSONDecodeError:
					continue

			except (socket.timeout, ConnectionError, BrokenPipeError) as e:
				logger.error(f"Connection error: {e}")

	def send_channels(self, payload: dict):
		frame = build_packet(payload)
		if frame is not None:
			self.packets.put(frame)
			return True

	def serial_loop(self):
		while not self.stop_request.is_set():
			try:
				if self.opened():
					event = self.packets.get(timeout=1.0)
					self.port.write(event)
					self.packets.task_done()
				elif self.serial_port_name is not None:
					self.open()

			except Empty:
				pass

			except SerialException as e:
				logger.error(f"Error opening or communicating with serial port: {e}")
				sleep(1)

		self.close() 
		logger.info("Stopped")
