#!/usr/bin/python3

import logging
import os
import serial
import serial.tools.list_ports
from .serialdev.device import Device

logger = logging.getLogger("coral")

def get_serial_ports():
	ports = dict()
	SYS_TTY_PATH = '/sys/class/tty'
	try:
		for entry in os.scandir(SYS_TTY_PATH):
			device_name = entry.name

			if device_name in ('.', '..'):
				continue

			dev_path = os.path.join('/dev', device_name)
			if not os.access(dev_path, os.R_OK | os.W_OK):
				continue
			if os.path.islink(dev_path):
				continue

			driver_path = os.path.join(SYS_TTY_PATH, device_name, 'device/driver')
			if not os.path.exists(driver_path):
				continue

			port = dict()

			if os.path.islink(driver_path):
				port['driver'] = os.path.basename(os.readlink(driver_path))

			type_path = os.path.join(SYS_TTY_PATH, device_name, 'type')
			if os.path.exists(type_path):
				port['fixed'] = True

			ports[dev_path] = port
	
	except Exception as e:
		print(e)

	comports = serial.tools.list_ports.comports(include_links=False)
	for device_name, desc, hwid in sorted(comports):
		port = ports.get(device_name)
		if port is not None:
			port['descrition'] = desc
			port['hwid'] = hwid
		else:
			ports[device_name] = dict(descrition = desc, hwid = hwid)

	return {i: ports[i] for i in sorted(ports)}

def find_devices():
	devices = dict()
	ports = get_serial_ports()
	for port in ports:
		#print(f"{port}: {ports.get(port)}")
		dev = Device(port)
		if dev.open():
			if dev.read(0, 3):
				logger.info(f"Найден {dev}")
				devices[port] = dict(name = dev.name, version = dev.version)
			else:
				dev.set_baudrate(420000)
				if dev.read(0, 3):
					logger.info(f"Найден {dev}")
					devices[port] = dict(name = dev.name, version = dev.version)
			dev.close()
	return devices

if __name__ == '__main__':
	print(find_devices())
