import logging
import time
from typing import List, Dict, Optional
import serial
from .register import Register, Type
from .packet import Packet, RequestRead, RequestName, RequestWrite, Response, Command

logger = logging.getLogger("coral.device")

class Device:
	
	def __init__(self, portname: str) -> None:
		self.port_name = portname
		self.name: str = None
		self.version: str = None
		self.port = serial.Serial()
		self.registers: Dict[int, Register] = dict()

	def open(self) -> bool:
		self.port.port = self.port_name
		self.port.baudrate = 115200
		self.port.bytesize = serial.EIGHTBITS
		self.port.parity = serial.PARITY_NONE
		self.port.stopbits = serial.STOPBITS_ONE
		self.port.timeout = 0.0
		self.port.inter_byte_timeout = 0.01
		self.port.xonxoff = False
		self.port.rtscts = False
		self.port.dsrdtr = False
		try: 
			self.port.open()
		except Exception as e:
			logger.error(f"Opening serial port {self.port_name}: {e}")
			return False
		self.port.flushInput()
		self.port.flushOutput()
		return True
	
	def set_baudrate(self, baudrate):
		if baudrate == self.port.baudrate:
			return True
		if self.opened():
			self.port.close()
		
		self.port.baudrate = baudrate
		try: 
			self.port.open()
		except Exception as e:
			logger.error(f"Opening serial port {self.port_name}: {e}")
			return False
		self.port.flushInput()
		self.port.flushOutput()
		return True

	def opened(self) -> bool:
		if self.port is not None:
			return self.port.is_open
		else:
			return False
	
	def close(self):
		if self.opened():
			self.port.close()
			self.port = None
	
	def request(self, req: Packet) -> Packet | None:
		if not self.opened():
			return None
		tx = req.to_bytes()
		try:
			time_tx = time.time()
			self.port.write(tx)
			del tx
			elapsed_tx = time.time() - time_tx
			rx = bytearray()
			time_rx = time.time()
			while True:
				rx += self.port.read(1024)
				elapsed_rx = time.time() - time_rx
				packet_start = None
				for index, byte in enumerate(rx):
					if (packet_start is None) and (byte == ord('<')):
						packet_start = index
					if ((packet_start is not None) and (byte == ord('>'))):						
						#print(f"Получено {len(rx)} байт за {elapsed_rx * 1000:.1f}мс")
						response = Response(rx[packet_start+1:index])
						if response.valid:
							if (response.command == req.command) and (response.start_address == req.address):
								return response
							if response.command == Command.ERROR:
								print(f"Отказ: {response.error}")
							return None
						packet_start = None
				if elapsed_rx > 0.1:
					print(f"Ответ не получен за {elapsed_rx * 1000:.1f}мс")
					return None
		except (OSError, serial.SerialException):
			pass
		return None
	
	def registers_merge(self, regs: List[Register]) -> int:
		count = 0
		for r in regs:
			if r.type != Type.EMPTY:
				count += 1
				if r.address in	self.registers:
					self.registers[r.address].update(r)
				else:
					self.registers[r.address] = r
					self.read_name(r.address)
				if r.address == 0:
					self.name = r.value
				elif r.address == 1:
					self.version = r.value
		return count
	
	def get(self, index):
		if type(index) is str:
			index = self.find(index)
		if type(index) is not int:
			return None
		if index in	self.registers:
			reg = self.registers[index]
			elapsed = time.time() - reg.updated
			if elapsed < 1.0:
				return reg.get()
			else:
				return self.read(index)
		return None
	
	def read(self, index: int, count: int = 1):
		req = RequestRead(index, count)
		response = self.request(req)
		del req
		if response is not None:
			self.registers_merge(response.registers)
			del response
			if count == 1:
				return self.registers[index].get()
			else:
				return True
		return None
	
	def read_all(self):
		index = 0
		while True:
			req = RequestRead(index, 10)
			response = self.request(req)
			del req
			if response is not None:
				count = self.registers_merge(response.registers)
				del response
				if count == 10:
					index += 10
				else:
					return index + count
			else:
				return index
	
	def read_name(self, index: int):
		req = RequestName(index)
		response = self.request(req)
		del req
		if response is not None:
			r = response.registers[0]
			name = None
			if r.address in	self.registers:
				name = self.registers[r.address].set_name(r.name)
			del response
			return name
		return None
	
	def write(self, index, value):
		if type(index) is str:
			index = self.find(index)
		if type(index) is not int:
			return False
		if index in	self.registers:
			reg = self.registers[index]
			reg.set(value)
			value = reg.value
			reg_type = reg.type
		else:
			reg_type = Type.DWORD
		req = RequestWrite(index, reg_type, value)
		self.registers[index].updated = 0
		response = self.request(req)
		del req
		if response is not None and response.start_address == index:
			del response
			return True
		return False

	def find(self, name: str) -> int:
		for index, reg in self.registers.items():
			if reg.name == name:
				return index
		return None

	def __getitem__(self, index):
		if type(index) is str:
			index = self.find(index)
		if type(index) is int:
			value = self.get(index)
			if value is not None:
				return value
			else:
				return self.read(index)
		return None

	def __setitem__(self, index, value):
		return self.write(index, value)

	def __repr__(self) -> str:
		return f"[{self.port_name}] {self.name} v{self.version}"
