from enum import Enum
from typing import List
from dataclasses import dataclass

class Type(Enum):
	EMPTY = 0
	BYTE = 1
	WORD = 2
	DWORD = 3
	INT = 4
	FLOAT = 5
	STRING = 6
	RAW = 7

@dataclass
class Register:
	address: int
	type: Type
	value = None
	name: str = None
	unit: str = None
	param: str = None
	range: str = None
	multiplier: float = None
	values_text: List[str] = None
	updated: float = float('nan')
	
	def __repr__(self) -> str:
		return f"[{self.address}] {self.type.name} = {self.value}"
	
	def __str__(self) -> str:
		if self.multiplier is not None:
			text = str(float(self.value) * self.multiplier)
		elif (self.values_text is not None) and (self.value < len(self.values_text)):
			text = f"[{self.value}] {self.values_text[self.value]}"
		else:
			text = str(self.value)
		if self.unit is not None:
			text += f" {self.unit}"
		return text
	
	def get(self):
		if self.value is None:
			return None
		elif self.multiplier is not None:
			return float(self.value) * self.multiplier
		elif (self.values_text is not None) and (self.value < len(self.values_text)):
			return self.values_text[self.value]
		else:		
			return self.value

	def set(self, value):
		if self.multiplier is not None:
			self.value = int(float(value) / self.multiplier)
		elif (self.values_text is not None) and (type(value) is str):
			self.value = self.values_text.index(value)
		elif self.type == Type.FLOAT:
			self.value = float(value)
		elif self.type == Type.STRING:
			self.value = str(value)
		else:
			self.value = int(value)
	
	def update(self, r):
		if type(r) is Register:
			self.value = r.value
			self.updated = r.updated
	
	def set_name(self, txt: str) -> str | None:
		# Выделяем из строки параметры значения
		txt = txt.split('/', 1)
		name = txt[0] if len(txt) > 0 else None
		# Выделяем из имени диапазон значений
		bracket_pos = name.rfind('(')
		if bracket_pos > 0:
			self.range = name[bracket_pos:]
			name = name[:bracket_pos].strip()
		# Выделяем из имени единицу измерения
		name = name.rsplit(',', 1)
		self.name = name[0] if len(name) > 0 else None
		self.unit = name[1].strip() if len(name) > 1 else None
		# Разбираем параметры значения
		if len(txt) > 1:
			self.param = txt[1]
			if (len(self.param) >= 2) and (self.param[0] == '*'):
				try:
					self.multiplier = float(self.param[1:])
				except ValueError:
					self.multiplier = 1.0
				self.values_text = None
			else:
				self.multiplier = None
				self.values_text = self.param.split(',')
		else:
			self.param = None
			self.multiplier = None
			self.values_text = None
		return self.name
