#!/usr/bin/python3
import logging
import time
import atexit
import shelve
from dataclasses import dataclass, asdict
from typing import Dict
from dacite import from_dict
import json

@dataclass
class Sync:
	test_count: int
	received: int
	damaged: int
	recovered: int
	quality: float

@dataclass
class Test:
	last_sync: int
	count: int
	result: Dict[int, Sync]

# Настраиваем логирование
logging.basicConfig(
	level=logging.DEBUG,
	format="%(asctime)s:%(levelname)s:%(name)s:%(message)s", 
	datefmt="%Y%m%d-%H%M%S"
)
logger = logging.getLogger("test")
logger.info("Start")

# Открываем хранилище
persistent_storage = shelve.open("sync")

# Начало тестирования
test = Test(0, 0, {})
try:
	result = persistent_storage.get("result")
	if result is not None:
		test = from_dict(data_class=Test, data=result)
		print(f"Накоплено результатов {test.count}, последний sync {test.last_sync:X}")
	else:
		print(f"Результатов нет")
except Exception as e:
	print(f"Ошибка чтения результатов: {e}")

# Корректное завершение при остановке
def on_shutdown():
	logger.info(f"Выход")
	persistent_storage.close()

atexit.register(on_shutdown)


best_count = 0
best_balanced = []
for sync, info in test.result.items():

	if info.quality <= 0.001:
		low_bit_count = (sync & 0xFF).bit_count()
		if low_bit_count == 4:
			best_count += 1
			best_balanced.append(sync)
			print(f"{sync:04X}: {info.quality:.3f}, {info.test_count} => {info.received} / {info.damaged}, {info.recovered}")


print(f"Завершено, результатов {best_count}")
best_balanced.sort()
with open('best_balanced.json', 'w') as json_file:
	json.dump(best_balanced, json_file, indent=4)
