#!/usr/bin/python3
import logging
import time
import atexit
import shelve
from dataclasses import dataclass, asdict
from typing import Dict
from dacite import from_dict
import json

@dataclass
class Sync:
	test_count: int
	received: int
	damaged: int
	recovered: int
	quality: float

@dataclass
class Test:
	last_sync: int
	count: int
	result: Dict[int, Sync]

# Настраиваем логирование
logging.basicConfig(
	level=logging.DEBUG,
	format="%(asctime)s:%(levelname)s:%(name)s:%(message)s", 
	datefmt="%Y%m%d-%H%M%S"
)
logger = logging.getLogger("test")
logger.info("Start")

# Открываем хранилище
persistent_storage = shelve.open("sync")

# Начало тестирования
test = Test(0, 0, {})
try:
	result = persistent_storage.get("result")
	if result is not None:
		test = from_dict(data_class=Test, data=result)
		print(f"Накоплено результатов {test.count}, последний sync {test.last_sync:X}")
	else:
		print(f"Результатов нет")
except Exception as e:
	print(f"Ошибка чтения результатов: {e}")

# Корректное завершение при остановке
def on_shutdown():
	logger.info(f"Выход")
	persistent_storage.close()

atexit.register(on_shutdown)


good_count = 0
good = []
for sync, info in test.result.items():

	if info.quality <= 0.01:
		good_count += 1
		good.append(dict(
			sync = sync,
			quality = 1.0 - info.quality,
		))
		print(f"{sync:04X}: {info.quality:.3f}, {info.test_count} => {info.received} / {info.damaged}, {info.recovered}")


print(f"Завершено, результатов {good_count}")
good = sorted(good, key=lambda x: x['sync'])
with open('good.json', 'w') as json_file:
    json.dump(good, json_file, indent=4)
