#!/usr/bin/python3
import logging
import time
import atexit
import shelve
from dataclasses import dataclass, asdict
from typing import Dict
from dacite import from_dict
from coral import Coral, find_devices

@dataclass
class Sync:
	test_count: int
	received: int
	damaged: int
	recovered: int
	quality: float

@dataclass
class Test:
	last_sync: int
	count: int
	result: Dict[int, Sync]

# Настраиваем логирование
logging.basicConfig(
	level=logging.DEBUG,
	format="%(asctime)s:%(levelname)s:%(name)s:%(message)s", 
	datefmt="%Y%m%d-%H%M%S"
)
logger = logging.getLogger("test")
logger.info("Start")

# Открываем хранилище
persistent_storage = shelve.open("sync.dat")

# Настраиваем Коралл
coral_devices = find_devices()
coral_tx = None
coral_rx = None

for port, device in coral_devices.items():
	if device.get("name") == "Коралл-15":
		coral_tx = Coral(port, None, None)
		logger.info(f"Найден передатчик: {port}")
	if device.get("name") == "Коралл-Б":
		coral_rx = Coral(port, None, None, 420000)
		logger.info(f"Найден приемник: {port}")
if coral_tx is None or coral_rx is None:
	print("Не найдены устройства")
	exit(1)

# Начало тестирования
good = []

test = Test(0, 0, {})
try:
	result = persistent_storage.get("result")
	if result is not None:
		test = from_dict(data_class=Test, data=result)
		print(f"Накоплено результатов {test.count}, последний sync {test.last_sync:X}")
	else:
		print(f"Результатов нет")
except Exception as e:
	print(f"Ошибка чтения результатов: {e}")

# Корректное завершение при остановке
def on_shutdown():
	logger.info(f"Выход, сохранение {test.count} результатов")
	persistent_storage["result"] = asdict(test)
	persistent_storage.close()

atexit.register(on_shutdown)

sync = 0xEE33
while sync < 0xFFFF:

	sync += 1
	low_bit_count = (sync & 0xFF).bit_count()
	high_bit_count = (sync >> 8).bit_count()
	if low_bit_count < 1 or low_bit_count > 7:
		continue
	if high_bit_count < 1 or high_bit_count > 7:
		continue
	if low_bit_count != 4:
		continue

	if coral_tx.control.write("Sync", sync) != True:
		print(f"Ошибка установки sync передатчика, sync {sync:X}")
		sync = last_sync
		continue
	if coral_rx.control.write("Sync", sync) != True:
		print(f"Ошибка установки sync приемника, sync {sync:X}")
		sync = last_sync
		continue
	time.sleep(2)

	sent = coral_tx["Пакетов отправлено"]
	received = coral_rx["Пакетов принято"]
	damaged = coral_rx["Пакетов повреждено"]
	recovered = coral_rx["Пакетов восстановлено"]
	
	if sent is None or received is None or damaged is None or recovered is None:
		sync = last_sync
		coral_tx.control.port.flushInput()
		coral_tx.control.port.flushOutput()
		coral_rx.control.port.flushInput()
		coral_rx.control.port.flushOutput()
		continue
	last_sync = sync

	if sent > 100:
		if sent - received > 20:
			quality = 1.0
		else:
			quality = float(damaged * 3 + recovered) / received
		test.last_sync = sync
		if quality < 0.2:
			saved = test.result.get(sync)
			if saved is None:
				test.count += 1
				test.result[sync] = Sync(sent, received, damaged, recovered, quality)
			else:
				saved.test_count += sent
				saved.received += received
				saved.damaged += damaged
				saved.recovered += recovered
				if saved.test_count - saved.received > saved.test_count * 0.1:
					saved.quality = 1.0
				else:
					saved.quality = float((saved.damaged * 3 + saved.recovered) / saved.received)
		print(f"{sync:04X}: {quality:.3f}, {sent} => {received} / {damaged}, {recovered}")
	else:
		print(f"err {sync:0X}: {sent} => {received} / {damaged}, {recovered}")


print(f"Завершено, результатов {test.count}")
